/*
Copyright IBM Corp. 2016 All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

		 http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package bccsp

const (
	//SM2 EC Digital Signature Algorithm(key gen, import, sign, verify)
	//SM2 EC Signature over GM256 curve y^3=x^3+ax+b.
	SM2 = "SM2"
	//SM2ReRand SM2 key re-randomization for keyderiv
	SM2ReRand = "SM2_RERAND"
	//SM3 Hash Algorithm
	SM3 = "SM3"
	//SM4 Sym Algorithm
	SM4 = "SM4"
)

//SM2KeyGenOpts: options for SM2 key generation.
type SM2KeyGenOpts struct {
	Temporary bool
}

//Algorithm returns the key generation algorithm identifier.
func (opts *SM2KeyGenOpts) Algorithm() string {
	return SM2
}

//Ephemeral returns true if the key to generate has to be ephemeral,
//false otherwise.
func (opts *SM2KeyGenOpts) Ephemeral() bool {
	return opts.Temporary
}

// SM2PrivateKeyImportOpts contains options for SM2 private key importation in DER or PKCS#8 format
type SM2PrivateKeyImportOpts struct {
	Temporary bool
}

// Algorithm returns the key importation algorithm identifier (to be used).
func (opts *SM2PrivateKeyImportOpts) Algorithm() string {
	return SM2
}

// Ephemeral returns true if the key to generate has to be ephemeral,
// false otherwise.
func (opts *SM2PrivateKeyImportOpts) Ephemeral() bool {
	return opts.Temporary
}

// SM2PublicKeyImportOpts contains options for SM2 key importation from sm2PublicKey
type SM2PublicKeyImportOpts struct {
	Temporary bool
}

// Algorithm returns the key importation algorithm identifier (to be used).
func (opts *SM2PublicKeyImportOpts) Algorithm() string {
	return SM2
}

// Ephemeral returns true if the key to generate has to be ephemeral,
// false otherwise.
func (opts *SM2PublicKeyImportOpts) Ephemeral() bool {
	return opts.Temporary
}

// SM2GoPublicKeyImportOpts contains options for SM2 key importation from sm2PublicKey
type SM2GoPublicKeyImportOpts struct {
	Temporary bool
}

// Algorithm returns the key importation algorithm identifier (to be used).
func (opts *SM2GoPublicKeyImportOpts) Algorithm() string {
	return SM2
}

// Ephemeral returns true if the key to generate has to be ephemeral,
// false otherwise.
func (opts *SM2GoPublicKeyImportOpts) Ephemeral() bool {
	return opts.Temporary
}

// SM2PKIXPublicKeyImportOpts contains options for SM2 key importation from sm2PublicKey
type SM2PKIXPublicKeyImportOpts struct {
	Temporary bool
}

// Algorithm returns the key importation algorithm identifier (to be used).
func (opts *SM2PKIXPublicKeyImportOpts) Algorithm() string {
	return SM2
}

// Ephemeral returns true if the key to generate has to be ephemeral,
// false otherwise.
func (opts *SM2PKIXPublicKeyImportOpts) Ephemeral() bool {
	return opts.Temporary
}

//SM2ReRandKeyOpts contains options for SM2 key re-randomization
type SM2ReRandKeyOpts struct {
	Temporary bool
	Expansion []byte
}

// Algorithm returns the key derivation algorithm identifier (to be used).
func (opts *SM2ReRandKeyOpts) Algorithm() string {
	return SM2ReRand
}

// Ephemeral returns true if the key to generate has to be ephemeral,
// false otherwise.
func (opts *SM2ReRandKeyOpts) Ephemeral() bool {
	return opts.Temporary
}

// ExpansionValue returns the re-randomization factor
func (opts *SM2ReRandKeyOpts) ExpansionValue() []byte {
	return opts.Expansion
}

//SM4KeyGenOpts contains options for SM4 key generation at 128 security level
type SM4KeyGenOpts struct {
	Temporary bool
}

//Algorithm returns the hash algorithm identifier(to be used)
func (opts *SM4KeyGenOpts) Algorithm() string {
	return SM4
}

//Ephemeral returns true if the key to generate has to be ephemeral,
//false otherwise
func (opts *SM4KeyGenOpts) Ephemeral() bool {
	return opts.Temporary
}

//SM4CBCPKCS7ModeOpts contains options for SM4 CBC encryption
//with PKCS7 padding
type SM4CBCPKCS7ModeOpts struct{}

//SM3Opts contains options relating to SM3
type SM3Opts struct {
}

//Algorithm returns the hash algorithm identifier(to be used)
func (opts *SM3Opts) Algorithm() string {
	return SM3
}
