/*
 * Decompiled with CFR 0.152.
 */
package org.hyperledger.fabric.sdk.security;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.math.BigInteger;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.Security;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.PKIXParameters;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPrivateKey;
import java.security.spec.ECGenParameterSpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.security.auth.x500.X500Principal;
import javax.xml.bind.DatatypeConverter;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.DERSequenceGenerator;
import org.bouncycastle.asn1.gm.GMObjectIdentifiers;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x9.ECNamedCurveTable;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.crypto.Digest;
import org.bouncycastle.crypto.digests.SHA256Digest;
import org.bouncycastle.crypto.digests.SHA3Digest;
import org.bouncycastle.crypto.digests.SM3Digest;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.pkcs.jcajce.JcaPKCS10CertificationRequestBuilder;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemObjectGenerator;
import org.bouncycastle.util.io.pem.PemReader;
import org.hyperledger.fabric.sdk.exception.CryptoException;
import org.hyperledger.fabric.sdk.exception.InvalidArgumentException;
import org.hyperledger.fabric.sdk.helper.Config;
import org.hyperledger.fabric.sdk.helper.DiagnosticFileDumper;
import org.hyperledger.fabric.sdk.helper.Utils;
import org.hyperledger.fabric.sdk.security.CryptoSuite;
import org.hyperledger.fabric.sdk.security.CryptoSuiteFactory;
import org.hyperledger.fabric.sdk.security.HLSDKJCryptoSuiteFactory;

public class CryptoPrimitives
implements CryptoSuite {
    private static final Log logger = LogFactory.getLog(CryptoPrimitives.class);
    private static final Config config = Config.getConfig();
    private static final boolean IS_TRACE_LEVEL = logger.isTraceEnabled();
    private static final String SM3WITHSM2 = "SM3WITHSM2";
    private static final DiagnosticFileDumper diagnosticFileDumper = IS_TRACE_LEVEL ? config.getDiagnosticFileDumper() : null;
    private String curveName;
    private CertificateFactory cf;
    private Provider SECURITY_PROVIDER;
    private String hashAlgorithm = config.getHashAlgorithm();
    private int securityLevel = config.getSecurityLevel();
    private String CERTIFICATE_FORMAT = config.getCertificateFormat();
    private String DEFAULT_SIGNATURE_ALGORITHM = config.getSignatureAlgorithm();
    private Map<Integer, String> securityCurveMapping = config.getSecurityCurveMapping();
    private KeyStore trustStore = null;
    final Set<String> certificateSet = ConcurrentHashMap.newKeySet();
    private final AtomicBoolean inited = new AtomicBoolean(false);

    public CryptoPrimitives() throws ClassNotFoundException, IllegalAccessException, InstantiationException {
        String securityProviderClassName = config.getSecurityProviderClassName();
        this.SECURITY_PROVIDER = this.setUpExplicitProvider(securityProviderClassName);
    }

    Provider setUpExplicitProvider(String securityProviderClassName) throws InstantiationException, ClassNotFoundException, IllegalAccessException {
        if (null == securityProviderClassName) {
            throw new InstantiationException(String.format("Security provider class name property (%s) set to null  ", "org.hyperledger.fabric.sdk.security_provider_class_name"));
        }
        if ("org.hyperledger.fabric.sdk.security.default_jdk_provider".equals(securityProviderClassName)) {
            return null;
        }
        Class<?> aClass = Class.forName(securityProviderClassName);
        if (null == aClass) {
            throw new InstantiationException(String.format("Getting class for security provider %s returned null  ", securityProviderClassName));
        }
        if (!Provider.class.isAssignableFrom(aClass)) {
            throw new InstantiationException(String.format("Class for security provider %s is not a Java security provider", aClass.getName()));
        }
        Provider securityProvider = (Provider)aClass.newInstance();
        if (securityProvider == null) {
            throw new InstantiationException(String.format("Creating instance of security %s returned null  ", aClass.getName()));
        }
        return securityProvider;
    }

    @Override
    public Certificate bytesToCertificate(byte[] certBytes) throws CryptoException {
        if (certBytes == null || certBytes.length == 0) {
            throw new CryptoException("bytesToCertificate: input null or zero length");
        }
        return this.getX509Certificate(certBytes);
    }

    private X509Certificate getX509Certificate(byte[] pemCertificate) throws CryptoException {
        X509Certificate ret = null;
        CryptoException rete = null;
        LinkedList<Provider> providerList = new LinkedList<Provider>(Arrays.asList(Security.getProviders()));
        if (this.SECURITY_PROVIDER != null) {
            providerList.add(this.SECURITY_PROVIDER);
        }
        try {
            providerList.add((Provider)BouncyCastleProvider.class.newInstance());
        }
        catch (Exception e) {
            logger.warn((Object)e);
        }
        for (Provider provider : providerList) {
            try {
                CertificateFactory certFactory;
                if (null == provider || null == (certFactory = CertificateFactory.getInstance(this.CERTIFICATE_FORMAT, provider))) continue;
                ByteArrayInputStream bis = new ByteArrayInputStream(pemCertificate);
                Throwable throwable = null;
                try {
                    Certificate certificate = certFactory.generateCertificate(bis);
                    if (!(certificate instanceof X509Certificate)) continue;
                    ret = (X509Certificate)certificate;
                    rete = null;
                    break;
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (bis == null) continue;
                    if (throwable != null) {
                        try {
                            bis.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    bis.close();
                }
            }
            catch (Exception e) {
                rete = new CryptoException(e.getMessage(), e);
            }
        }
        if (null != rete) {
            throw rete;
        }
        if (ret == null) {
            logger.error((Object)"Could not convert pem bytes");
        }
        return ret;
    }

    public PrivateKey bytesToPrivateKey(byte[] pemKey) throws CryptoException {
        PrivateKey pk;
        try {
            PemReader pr = new PemReader((Reader)new StringReader(new String(pemKey)));
            PemObject po = pr.readPemObject();
            PEMParser pem = new PEMParser((Reader)new StringReader(new String(pemKey)));
            if (po.getType().equals("PRIVATE KEY")) {
                pk = new JcaPEMKeyConverter().getPrivateKey((PrivateKeyInfo)pem.readObject());
            } else {
                logger.trace((Object)("Found private key with type " + po.getType()));
                PEMKeyPair kp = (PEMKeyPair)pem.readObject();
                pk = new JcaPEMKeyConverter().getPrivateKey(kp.getPrivateKeyInfo());
            }
        }
        catch (Exception e) {
            throw new CryptoException("Failed to convert private key bytes", e);
        }
        return pk;
    }

    @Override
    public boolean verify(byte[] pemCertificate, String signatureAlgorithm, byte[] signature, byte[] plainText) throws CryptoException {
        boolean isVerified = false;
        if (plainText == null || signature == null || pemCertificate == null) {
            return false;
        }
        if (config.extraLogLevel(10) && null != diagnosticFileDumper) {
            String message = "plaintext in hex: " + DatatypeConverter.printHexBinary((byte[])plainText) + '\n' + "signature in hex: " + DatatypeConverter.printHexBinary((byte[])signature) + '\n' + "PEM cert in hex: " + DatatypeConverter.printHexBinary((byte[])pemCertificate);
            logger.trace((Object)("verify :  " + diagnosticFileDumper.createDiagnosticFile(message)));
        }
        try {
            X509Certificate certificate = this.getX509Certificate(pemCertificate);
            if (certificate != null && (isVerified = this.validateCertificate(certificate))) {
                Signature sig = Signature.getInstance(certificate.getSigAlgName());
                sig.initVerify(certificate);
                sig.update(plainText);
                isVerified = sig.verify(signature);
            }
        }
        catch (InvalidKeyException e) {
            CryptoException ex = new CryptoException("Cannot verify signature. Error is: " + e.getMessage() + "\r\nCertificate: " + DatatypeConverter.printHexBinary((byte[])pemCertificate), e);
            logger.error((Object)ex.getMessage(), (Throwable)ex);
            throw ex;
        }
        catch (NoSuchAlgorithmException | SignatureException e) {
            CryptoException ex = new CryptoException("Cannot verify. Signature algorithm is invalid. Error is: " + e.getMessage(), e);
            logger.error((Object)ex.getMessage(), (Throwable)ex);
            throw ex;
        }
        return isVerified;
    }

    private void createTrustStore() throws CryptoException {
        try {
            KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
            keyStore.load(null, null);
            this.setTrustStore(keyStore);
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException | InvalidArgumentException e) {
            throw new CryptoException("Cannot create trust store. Error: " + e.getMessage(), e);
        }
    }

    private void setTrustStore(KeyStore keyStore) throws InvalidArgumentException {
        if (keyStore == null) {
            throw new InvalidArgumentException("Need to specify a java.security.KeyStore input parameter");
        }
        this.trustStore = keyStore;
    }

    public KeyStore getTrustStore() throws CryptoException {
        if (this.trustStore == null) {
            this.createTrustStore();
        }
        return this.trustStore;
    }

    public void addCACertificateToTrustStore(File caCertPem, String alias) throws CryptoException, InvalidArgumentException {
        if (caCertPem == null) {
            throw new InvalidArgumentException("The certificate cannot be null");
        }
        if (alias == null || alias.isEmpty()) {
            throw new InvalidArgumentException("You must assign an alias to a certificate when adding to the trust store");
        }
        try (BufferedInputStream bis = new BufferedInputStream(new ByteArrayInputStream(FileUtils.readFileToByteArray((File)caCertPem)));){
            Certificate caCert = this.cf.generateCertificate(bis);
            this.addCACertificateToTrustStore(caCert, alias);
        }
        catch (IOException | CertificateException e) {
            throw new CryptoException("Unable to add CA certificate to trust store. Error: " + e.getMessage(), e);
        }
    }

    public void addCACertificatesToTrustStore(BufferedInputStream bis) throws CryptoException, InvalidArgumentException {
        if (bis == null) {
            throw new InvalidArgumentException("The certificate stream bis cannot be null");
        }
        try {
            Collection<? extends Certificate> certificates = this.cf.generateCertificates(bis);
            for (Certificate certificate : certificates) {
                this.addCACertificateToTrustStore(certificate);
            }
        }
        catch (CertificateException e) {
            throw new CryptoException("Unable to add CA certificate to trust store. Error: " + e.getMessage(), e);
        }
    }

    private void addCACertificateToTrustStore(Certificate certificate) throws InvalidArgumentException, CryptoException {
        String alias = certificate instanceof X509Certificate ? ((X509Certificate)certificate).getSerialNumber().toString() : Integer.toString(certificate.hashCode());
        this.addCACertificateToTrustStore(certificate, alias);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void addCACertificateToTrustStore(Certificate caCert, String alias) throws InvalidArgumentException, CryptoException {
        if (alias == null || alias.isEmpty()) {
            throw new InvalidArgumentException("You must assign an alias to a certificate when adding to the trust store.");
        }
        if (caCert == null) {
            throw new InvalidArgumentException("Certificate cannot be null.");
        }
        try {
            if (config.extraLogLevel(10) && null != diagnosticFileDumper) {
                logger.trace((Object)(String.format("Adding cert to trust store. alias: %s. certificate:", alias) + diagnosticFileDumper.createDiagnosticFile(alias + "cert: " + caCert.toString())));
            }
            Set<String> set = this.certificateSet;
            synchronized (set) {
                if (this.certificateSet.contains(alias)) {
                    return;
                }
                this.getTrustStore().setCertificateEntry(alias, caCert);
                this.certificateSet.add(alias);
            }
        }
        catch (KeyStoreException e) {
            String emsg = "Unable to add CA certificate to trust store. Error: " + e.getMessage();
            logger.error((Object)emsg, (Throwable)e);
            throw new CryptoException(emsg, e);
        }
    }

    @Override
    public void loadCACertificates(Collection<Certificate> certificates) throws CryptoException {
        if (certificates == null || certificates.size() == 0) {
            throw new CryptoException("Unable to load CA certificates. List is empty");
        }
        try {
            for (Certificate cert : certificates) {
                this.addCACertificateToTrustStore(cert);
            }
        }
        catch (InvalidArgumentException e) {
            throw new CryptoException("Unable to add certificate to trust store. Error: " + e.getMessage(), e);
        }
    }

    @Override
    public void loadCACertificatesAsBytes(Collection<byte[]> certificatesBytes) throws CryptoException {
        if (certificatesBytes == null || certificatesBytes.size() == 0) {
            throw new CryptoException("List of CA certificates is empty. Nothing to load.");
        }
        ArrayList<Certificate> certList = new ArrayList<Certificate>();
        for (byte[] certBytes : certificatesBytes) {
            certList.add(this.bytesToCertificate(certBytes));
        }
        this.loadCACertificates(certList);
    }

    public void addClientCACertificateToTrustStore(byte[] clientKey, byte[] clientCert, String clientKeyPassword) throws CryptoException, IllegalArgumentException {
        if (clientKey == null) {
            throw new IllegalArgumentException("Client key byte input stream is required.");
        }
        if (clientCert == null) {
            throw new IllegalArgumentException("Client certificate byte input stream is required.");
        }
        try {
            Certificate tlsClientCertificate = this.bytesToCertificate(clientCert);
            String alias = tlsClientCertificate instanceof X509Certificate ? ((X509Certificate)tlsClientCertificate).getSerialNumber().toString() : Integer.toString(tlsClientCertificate.hashCode());
            char[] password = clientKeyPassword == null ? new char[]{} : clientKeyPassword.toCharArray();
            this.getTrustStore().setKeyEntry(alias, this.bytesToPrivateKey(clientKey), password, new Certificate[]{tlsClientCertificate});
        }
        catch (KeyStoreException e) {
            throw new CryptoException("Unable to add client CA certificate to trust store.", e);
        }
    }

    boolean validateCertificate(byte[] certPEM) {
        if (certPEM == null) {
            return false;
        }
        try {
            X509Certificate certificate = this.getX509Certificate(certPEM);
            if (null == certificate) {
                throw new Exception("Certificate transformation returned null");
            }
            return this.validateCertificate(certificate);
        }
        catch (Exception e) {
            logger.error((Object)("Cannot validate certificate. Error is: " + e.getMessage() + "\r\nCertificate (PEM, hex): " + DatatypeConverter.printHexBinary((byte[])certPEM)));
            return false;
        }
    }

    boolean validateCertificate(Certificate cert) {
        boolean isValidated;
        if (cert == null) {
            return false;
        }
        try {
            KeyStore keyStore = this.getTrustStore();
            PKIXParameters parms = new PKIXParameters(keyStore);
            parms.setRevocationEnabled(false);
            CertPathValidator certValidator = CertPathValidator.getInstance(CertPathValidator.getDefaultType());
            ArrayList<Certificate> start = new ArrayList<Certificate>();
            start.add(cert);
            CertificateFactory certFactory = CertificateFactory.getInstance(this.CERTIFICATE_FORMAT);
            CertPath certPath = certFactory.generateCertPath(start);
            certValidator.validate(certPath, parms);
            isValidated = true;
        }
        catch (InvalidAlgorithmParameterException | KeyStoreException | NoSuchAlgorithmException | CertPathValidatorException | CertificateException | CryptoException e) {
            logger.error((Object)("Cannot validate certificate. Error is: " + e.getMessage() + "\r\nCertificate" + cert.toString()));
            isValidated = false;
        }
        return isValidated;
    }

    void setSecurityLevel(int securityLevel) throws InvalidArgumentException {
        logger.trace((Object)String.format("setSecurityLevel to %d", securityLevel));
        if (this.securityCurveMapping.isEmpty()) {
            throw new InvalidArgumentException("Security curve mapping has no entries.");
        }
        if (!this.securityCurveMapping.containsKey(securityLevel)) {
            StringBuilder sb = new StringBuilder();
            String sp = "";
            for (int x : this.securityCurveMapping.keySet()) {
                sb.append(sp).append(x);
                sp = ", ";
            }
            throw new InvalidArgumentException(String.format("Illegal security level: %d. Valid values are: %s", securityLevel, sb.toString()));
        }
        String lcurveName = this.securityCurveMapping.get(securityLevel);
        logger.debug((Object)String.format("Mapped curve strength %d to %s", securityLevel, lcurveName));
        X9ECParameters params = ECNamedCurveTable.getByName((String)lcurveName);
        if (params == null) {
            InvalidArgumentException invalidArgumentException = new InvalidArgumentException(String.format("Curve %s defined for security strength %d was not found.", this.curveName, securityLevel));
            logger.error((Object)invalidArgumentException);
            throw invalidArgumentException;
        }
        this.curveName = lcurveName;
        this.securityLevel = securityLevel;
    }

    void setHashAlgorithm(String algorithm) throws InvalidArgumentException {
        if (Utils.isNullOrEmpty(algorithm) || !"SHA2".equals(algorithm) && !"SHA3".equals(algorithm)) {
            throw new InvalidArgumentException("Illegal Hash function family: " + algorithm + " - must be either SHA2 or SHA3");
        }
        this.hashAlgorithm = algorithm;
    }

    @Override
    public KeyPair keyGen() throws CryptoException {
        return this.ecdsaKeyGen();
    }

    private KeyPair ecdsaKeyGen() throws CryptoException {
        return this.generateKey("EC", this.curveName);
    }

    private KeyPair generateKey(String encryptionName, String curveName) throws CryptoException {
        try {
            ECGenParameterSpec ecGenSpec = new ECGenParameterSpec(curveName);
            KeyPairGenerator g = this.SECURITY_PROVIDER == null ? KeyPairGenerator.getInstance(encryptionName) : KeyPairGenerator.getInstance(encryptionName, this.SECURITY_PROVIDER);
            g.initialize(ecGenSpec, new SecureRandom());
            return g.generateKeyPair();
        }
        catch (Exception exp) {
            throw new CryptoException("Unable to generate key pair", exp);
        }
    }

    private static BigInteger[] decodeECDSASignature(byte[] signature) throws Exception {
        try (ByteArrayInputStream inStream = new ByteArrayInputStream(signature);){
            ASN1InputStream asnInputStream = new ASN1InputStream((InputStream)inStream);
            ASN1Primitive asn1 = asnInputStream.readObject();
            BigInteger[] sigs = new BigInteger[2];
            int count = 0;
            if (asn1 instanceof ASN1Sequence) {
                ASN1Encodable[] asn1Encodables;
                ASN1Sequence asn1Sequence = (ASN1Sequence)asn1;
                for (ASN1Encodable asn1Encodable : asn1Encodables = asn1Sequence.toArray()) {
                    ASN1Primitive asn1Primitive = asn1Encodable.toASN1Primitive();
                    if (!(asn1Primitive instanceof ASN1Integer)) continue;
                    ASN1Integer asn1Integer = (ASN1Integer)asn1Primitive;
                    BigInteger integer = asn1Integer.getValue();
                    if (count < 2) {
                        sigs[count] = integer;
                    }
                    ++count;
                }
            }
            if (count != 2) {
                throw new CryptoException(String.format("Invalid ECDSA signature. Expected count of 2 but got: %d. Signature is: %s", count, DatatypeConverter.printHexBinary((byte[])signature)));
            }
            BigInteger[] bigIntegerArray = sigs;
            return bigIntegerArray;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private byte[] ecdsaSignToBytes(ECPrivateKey privateKey, byte[] data) throws CryptoException {
        if (data == null) {
            throw new CryptoException("Data that to be signed is null.");
        }
        if (data.length == 0) {
            throw new CryptoException("Data to be signed was empty.");
        }
        try {
            X9ECParameters params = ECNamedCurveTable.getByName((String)this.curveName);
            BigInteger curveN = params.getN();
            Signature sig = this.SECURITY_PROVIDER == null ? Signature.getInstance(this.DEFAULT_SIGNATURE_ALGORITHM) : Signature.getInstance(this.DEFAULT_SIGNATURE_ALGORITHM, this.SECURITY_PROVIDER);
            sig.initSign(privateKey);
            sig.update(data);
            byte[] signature = sig.sign();
            if (GMObjectIdentifiers.sm2sign_with_sm3.toString().equals(this.DEFAULT_SIGNATURE_ALGORITHM)) return signature;
            if (SM3WITHSM2.equals(this.DEFAULT_SIGNATURE_ALGORITHM.toUpperCase())) {
                return signature;
            }
            BigInteger[] sigs = this.preventMalleability(CryptoPrimitives.decodeECDSASignature(signature), curveN);
            try (ByteArrayOutputStream s = new ByteArrayOutputStream();){
                DERSequenceGenerator seq = new DERSequenceGenerator((OutputStream)s);
                seq.addObject((ASN1Encodable)new ASN1Integer(sigs[0]));
                seq.addObject((ASN1Encodable)new ASN1Integer(sigs[1]));
                seq.close();
                byte[] byArray = s.toByteArray();
                return byArray;
            }
        }
        catch (Exception e) {
            throw new CryptoException("Could not sign the message using private key", e);
        }
    }

    @Override
    public byte[] sign(PrivateKey key, byte[] data) throws CryptoException {
        return this.ecdsaSignToBytes((ECPrivateKey)key, data);
    }

    private BigInteger[] preventMalleability(BigInteger[] sigs, BigInteger curveN) {
        BigInteger sval = sigs[1];
        BigInteger cmpVal = curveN.divide(BigInteger.valueOf(2L));
        if (sval.compareTo(cmpVal) > 0) {
            sigs[1] = curveN.subtract(sval);
        }
        return sigs;
    }

    @Override
    public String generateCertificationRequest(String subject, KeyPair pair) throws InvalidArgumentException {
        try {
            JcaPKCS10CertificationRequestBuilder p10Builder = new JcaPKCS10CertificationRequestBuilder(new X500Principal("CN=" + subject), pair.getPublic());
            JcaContentSignerBuilder csBuilder = new JcaContentSignerBuilder("SHA256withECDSA");
            if (null != this.SECURITY_PROVIDER) {
                csBuilder.setProvider(this.SECURITY_PROVIDER);
            }
            ContentSigner signer = csBuilder.build(pair.getPrivate());
            return this.certificationRequestToPEM(p10Builder.build(signer));
        }
        catch (Exception e) {
            logger.error((Object)e);
            throw new InvalidArgumentException(e);
        }
    }

    private String certificationRequestToPEM(PKCS10CertificationRequest csr) throws IOException {
        PemObject pemCSR = new PemObject("CERTIFICATE REQUEST", csr.getEncoded());
        StringWriter str = new StringWriter();
        JcaPEMWriter pemWriter = new JcaPEMWriter((Writer)str);
        pemWriter.writeObject((PemObjectGenerator)pemCSR);
        pemWriter.close();
        str.close();
        return str.toString();
    }

    @Override
    public byte[] hash(byte[] input) {
        Digest digest = this.getHashDigest();
        byte[] retValue = new byte[digest.getDigestSize()];
        digest.update(input, 0, input.length);
        digest.doFinal(retValue, 0);
        return retValue;
    }

    @Override
    public CryptoSuiteFactory getCryptoSuiteFactory() {
        return HLSDKJCryptoSuiteFactory.instance();
    }

    public void init() throws CryptoException, InvalidArgumentException {
        if (this.inited.getAndSet(true)) {
            throw new InvalidArgumentException("Crypto suite already initialized");
        }
        this.resetConfiguration();
    }

    private Digest getHashDigest() {
        if ("SHA3".equals(this.hashAlgorithm)) {
            return new SHA3Digest();
        }
        if ("SM3".equals(this.hashAlgorithm)) {
            return new SM3Digest();
        }
        return new SHA256Digest();
    }

    private void resetConfiguration() throws CryptoException, InvalidArgumentException {
        this.setSecurityLevel(this.securityLevel);
        this.setHashAlgorithm(this.hashAlgorithm);
        try {
            this.cf = CertificateFactory.getInstance(this.CERTIFICATE_FORMAT);
        }
        catch (CertificateException e) {
            CryptoException ex = new CryptoException("Cannot initialize " + this.CERTIFICATE_FORMAT + " certificate factory. Error = " + e.getMessage(), e);
            logger.error((Object)ex.getMessage(), (Throwable)ex);
            throw ex;
        }
    }

    void setProperties(Properties properties) throws CryptoException, InvalidArgumentException {
        if (properties == null) {
            throw new InvalidArgumentException("properties must not be null");
        }
        this.hashAlgorithm = Optional.ofNullable(properties.getProperty("org.hyperledger.fabric.sdk.hash_algorithm")).orElse(this.hashAlgorithm);
        String secLevel = Optional.ofNullable(properties.getProperty("org.hyperledger.fabric.sdk.security_level")).orElse(Integer.toString(this.securityLevel));
        this.securityLevel = Integer.parseInt(secLevel);
        this.securityCurveMapping = properties.containsKey("org.hyperledger.fabric.sdk.security_curve_mapping") ? Config.parseSecurityCurveMappings(properties.getProperty("org.hyperledger.fabric.sdk.security_curve_mapping")) : config.getSecurityCurveMapping();
        String providerName = properties.containsKey("org.hyperledger.fabric.sdk.security_provider_class_name") ? properties.getProperty("org.hyperledger.fabric.sdk.security_provider_class_name") : config.getSecurityProviderClassName();
        try {
            this.SECURITY_PROVIDER = this.setUpExplicitProvider(providerName);
        }
        catch (Exception e) {
            throw new InvalidArgumentException(String.format("Getting provider for class name: %s", providerName), e);
        }
        this.CERTIFICATE_FORMAT = Optional.ofNullable(properties.getProperty("org.hyperledger.fabric.sdk.crypto.certificate_format")).orElse(this.CERTIFICATE_FORMAT);
        this.DEFAULT_SIGNATURE_ALGORITHM = Optional.ofNullable(properties.getProperty("org.hyperledger.fabric.sdk.crypto.default_signature_algorithm")).orElse(this.DEFAULT_SIGNATURE_ALGORITHM);
        this.resetConfiguration();
    }

    @Override
    public Properties getProperties() {
        Properties properties = new Properties();
        properties.setProperty("org.hyperledger.fabric.sdk.hash_algorithm", this.hashAlgorithm);
        properties.setProperty("org.hyperledger.fabric.sdk.security_level", Integer.toString(this.securityLevel));
        properties.setProperty("org.hyperledger.fabric.sdk.crypto.certificate_format", this.CERTIFICATE_FORMAT);
        properties.setProperty("org.hyperledger.fabric.sdk.crypto.default_signature_algorithm", this.DEFAULT_SIGNATURE_ALGORITHM);
        return properties;
    }

    public byte[] certificateToDER(String certificatePEM) {
        byte[] content = null;
        try (PemReader pemReader = new PemReader((Reader)new StringReader(certificatePEM));){
            PemObject pemObject = pemReader.readPemObject();
            content = pemObject.getContent();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return content;
    }
}

